
(** Output for HTML combinators.

    This output type is used to produce a string reasonably efficiently from
    a tree of combinators.

    {b NOTE}: this is experimental and an unstable API.

    @since 0.12
    @open *)
module Out : sig
  type t
  val create_of_buffer : Buffer.t -> t
  val create_of_out: IO.Output.t -> t
  val flush : t -> unit
  val add_char : t -> char -> unit
  val add_string : t -> string -> unit
  val add_format_nl : t -> unit
  val with_no_format_nl : t -> (unit -> 'a) -> 'a
end = struct
  type t = {
    out: IO.Output.t;
    mutable fmt_nl: bool; (* if true, we print [\n] around tags to format the html *)
  }
  let create_of_out out = {out; fmt_nl=true}
  let create_of_buffer buf : t = create_of_out (IO.Output.of_buffer buf)
  let[@inline] flush self : unit = IO.Output.flush self.out
  let[@inline] add_char self c = IO.Output.output_char self.out c
  let[@inline] add_string self s = IO.Output.output_string self.out s
  let[@inline] add_format_nl self = if self.fmt_nl then add_char self '\n'
  let with_no_format_nl self f =
    if self.fmt_nl then (
      self.fmt_nl <- false;
      try let x=f() in self.fmt_nl <- true; x with e -> self.fmt_nl <- true; raise e
    ) else f()
end

type attribute = string * string
(** An attribute, i.e. a key/value pair *)

type elt = Out.t -> unit
(** A html element. It is represented by its output function, so we
    can directly print it. *)

type void = ?if_:bool -> attribute list -> elt
(** Element without children. *)

type nary = ?if_:bool -> attribute list -> elt list -> elt
(** Element with children, represented as a list.
    @param if_ if false, do not print anything (default true) *)

(** A chunk of sub-elements, possibly empty.
    @inline *)
type sub_elt = [ `E of elt | `L of elt list | `S of elt Seq.t | `Nil]

type nary' = ?if_:bool -> attribute list -> sub_elt list -> elt
(** Element with children, represented as a list of {!sub_elt} to be flattened
    @param if_ if false, do not print anything (default true) *)

(**/**)
module Helpers_ = struct

(** Escape string so it can be safely embedded in HTML text. *)
let _str_escape (out:Out.t) (s:string) : unit =
  String.iter (function
    | '<' -> Out.add_string out "&lt;"
    | '>' -> Out.add_string out "&gt;"
    | '&' -> Out.add_string out "&amp;"
    | '"' -> Out.add_string out "&quot;"
    | '\'' -> Out.add_string out "&apos;"
    | c -> Out.add_char out c)
  s

(** Print the value part of an attribute *)
let _attr_escape (out:Out.t) (s:string) =
  Out.add_char out '"';
  _str_escape out s;
  Out.add_char out '"'

(** Output a list of attributes. *)
let _write_attrs (out:Out.t) (l:attribute list) : unit =
  List.iter
    (fun (k,v) ->
      Out.add_char out ' ';
      Out.add_string out k;
      Out.add_char out '=';
      _attr_escape out v)
    l

(** Write sub-elements of a {!nary'} element, returns [true] iff
    at least one sub-element was written. *)
let _write_subs (out:Out.t) (l:sub_elt list) : bool =
  let has_sub = ref false in
  let prepend_white () = has_sub := true; Out.add_format_nl out; in
  let emit1 = function
    | `E x -> prepend_white(); x out
    | `L l -> List.iter (fun e -> prepend_white(); e out) l
    | `S l -> Seq.iter (fun e -> prepend_white(); e out) l
    | `Nil -> ()
  in
  List.iter emit1 l;
  !has_sub

(** Write a tag, with its attributes.
    @param void if true, end with "/>", otherwise end with ">" *)
let _write_tag_attrs ~void (out:Out.t) (tag:string) (attrs:attribute list) : unit =
  Out.add_string out "<";
  Out.add_string out tag;
  _write_attrs out attrs;
  if void then Out.add_string out "/>" else Out.add_string out ">"

end
open Helpers_
(**/**)

(** Sub-element with a single element inside. *)
let[@inline] sub_e (elt:elt) : sub_elt = `E elt

(** Sub-element with a list of items inside. *)
let[@inline] sub_l (l:elt list) : sub_elt = `L l

(** Sub-element with a sequence ({!Seq.t}) of items inside. *)
let[@inline] sub_seq (l:elt Seq.t) : sub_elt = `S l

(** Helper to build a {!Seq.t} from an array. *)
let seq_of_array (a:_ array) : _ Seq.t =
  let rec loop i () =
    if i=Array.length a then Seq.Nil
    else Seq.Cons (a.(i), loop (i+1))
  in loop 0

(** Sub-element with nothing inside. Useful in conditionals, when one
    decides not to emit a sub-element at all. *)
let sub_empty : sub_elt = `Nil

(** Emit a string value, which will be escaped. *)
let txt (txt:string) : elt = fun out -> _str_escape out txt

(** Formatted version of {!txt} *)
let txtf fmt = Format.kasprintf (fun s -> fun out -> _str_escape out s) fmt

(** Emit raw HTML. Caution, this can lead to injection vulnerabilities,
  never use with text that comes from untrusted users. *)
let raw_html (s:string) : elt = fun out -> Out.add_string out s

(** tag "area", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area} mdn} *)
let area : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "area" attrs;
  )

(** tag "base", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/base} mdn} *)
let base : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "base" attrs;
  )

(** tag "br", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/br} mdn} *)
let br : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "br" attrs;
  )

(** tag "col", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/col} mdn} *)
let col : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "col" attrs;
  )

(** tag "embed", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/embed} mdn} *)
let embed : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "embed" attrs;
  )

(** tag "hr", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr} mdn} *)
let hr : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "hr" attrs;
  )

(** tag "img", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img} mdn} *)
let img : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "img" attrs;
  )

(** tag "input", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input} mdn} *)
let input : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "input" attrs;
  )

(** tag "link", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link} mdn} *)
let link : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "link" attrs;
  )

(** tag "menuitem", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/menuitem} mdn} *)
let menuitem : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "menuitem" attrs;
  )

(** tag "meta", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meta} mdn} *)
let meta : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "meta" attrs;
  )

(** tag "param", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/param} mdn} *)
let param : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "param" attrs;
  )

(** tag "source", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/source} mdn} *)
let source : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "source" attrs;
  )

(** tag "track", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track} mdn} *)
let track : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "track" attrs;
  )

(** tag "wbr", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/wbr} mdn} *)
let wbr : void = fun ?(if_=true) attrs out ->
  if if_ then (
    _write_tag_attrs ~void:true out "wbr" attrs;
  )

(** tag "a", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a} mdn} *)
let a : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "a" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</a>")

(** tag "a", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a} mdn} *)
let a' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "a" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</a>")

(** tag "abbr", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/abbr} mdn} *)
let abbr : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "abbr" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</abbr>")

(** tag "abbr", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/abbr} mdn} *)
let abbr' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "abbr" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</abbr>")

(** tag "address", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/address} mdn} *)
let address : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "address" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</address>")

(** tag "address", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/address} mdn} *)
let address' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "address" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</address>")

(** tag "article", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/article} mdn} *)
let article : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "article" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</article>")

(** tag "article", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/article} mdn} *)
let article' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "article" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</article>")

(** tag "aside", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/aside} mdn} *)
let aside : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "aside" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</aside>")

(** tag "aside", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/aside} mdn} *)
let aside' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "aside" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</aside>")

(** tag "audio", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio} mdn} *)
let audio : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "audio" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</audio>")

(** tag "audio", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio} mdn} *)
let audio' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "audio" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</audio>")

(** tag "b", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/b} mdn} *)
let b : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "b" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</b>")

(** tag "b", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/b} mdn} *)
let b' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "b" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</b>")

(** tag "bdi", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bdi} mdn} *)
let bdi : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "bdi" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</bdi>")

(** tag "bdi", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bdi} mdn} *)
let bdi' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "bdi" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</bdi>")

(** tag "bdo", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bdo} mdn} *)
let bdo : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "bdo" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</bdo>")

(** tag "bdo", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bdo} mdn} *)
let bdo' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "bdo" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</bdo>")

(** tag "blockquote", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/blockquote} mdn} *)
let blockquote : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "blockquote" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</blockquote>")

(** tag "blockquote", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/blockquote} mdn} *)
let blockquote' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "blockquote" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</blockquote>")

(** tag "body", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body} mdn} *)
let body : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "body" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</body>")

(** tag "body", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body} mdn} *)
let body' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "body" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</body>")

(** tag "button", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button} mdn} *)
let button : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "button" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</button>")

(** tag "button", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button} mdn} *)
let button' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "button" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</button>")

(** tag "canvas", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/canvas} mdn} *)
let canvas : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "canvas" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</canvas>")

(** tag "canvas", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/canvas} mdn} *)
let canvas' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "canvas" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</canvas>")

(** tag "caption", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/caption} mdn} *)
let caption : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "caption" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</caption>")

(** tag "caption", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/caption} mdn} *)
let caption' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "caption" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</caption>")

(** tag "cite", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/cite} mdn} *)
let cite : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "cite" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</cite>")

(** tag "cite", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/cite} mdn} *)
let cite' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "cite" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</cite>")

(** tag "code", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/code} mdn} *)
let code : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "code" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</code>")

(** tag "code", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/code} mdn} *)
let code' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "code" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</code>")

(** tag "colgroup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/colgroup} mdn} *)
let colgroup : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "colgroup" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</colgroup>")

(** tag "colgroup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/colgroup} mdn} *)
let colgroup' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "colgroup" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</colgroup>")

(** tag "data", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/data} mdn} *)
let data : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "data" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</data>")

(** tag "data", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/data} mdn} *)
let data' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "data" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</data>")

(** tag "datalist", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/datalist} mdn} *)
let datalist : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "datalist" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</datalist>")

(** tag "datalist", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/datalist} mdn} *)
let datalist' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "datalist" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</datalist>")

(** tag "dd", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dd} mdn} *)
let dd : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dd" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</dd>")

(** tag "dd", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dd} mdn} *)
let dd' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dd" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</dd>")

(** tag "del", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/del} mdn} *)
let del : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "del" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</del>")

(** tag "del", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/del} mdn} *)
let del' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "del" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</del>")

(** tag "details", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/details} mdn} *)
let details : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "details" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</details>")

(** tag "details", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/details} mdn} *)
let details' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "details" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</details>")

(** tag "dfn", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dfn} mdn} *)
let dfn : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dfn" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</dfn>")

(** tag "dfn", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dfn} mdn} *)
let dfn' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dfn" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</dfn>")

(** tag "dialog", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dialog} mdn} *)
let dialog : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dialog" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</dialog>")

(** tag "dialog", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dialog} mdn} *)
let dialog' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dialog" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</dialog>")

(** tag "div", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/div} mdn} *)
let div : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "div" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</div>")

(** tag "div", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/div} mdn} *)
let div' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "div" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</div>")

(** tag "dl", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dl} mdn} *)
let dl : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dl" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</dl>")

(** tag "dl", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dl} mdn} *)
let dl' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dl" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</dl>")

(** tag "dt", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dt} mdn} *)
let dt : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dt" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</dt>")

(** tag "dt", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dt} mdn} *)
let dt' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "dt" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</dt>")

(** tag "em", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/em} mdn} *)
let em : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "em" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</em>")

(** tag "em", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/em} mdn} *)
let em' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "em" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</em>")

(** tag "fieldset", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/fieldset} mdn} *)
let fieldset : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "fieldset" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</fieldset>")

(** tag "fieldset", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/fieldset} mdn} *)
let fieldset' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "fieldset" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</fieldset>")

(** tag "figcaption", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/figcaption} mdn} *)
let figcaption : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "figcaption" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</figcaption>")

(** tag "figcaption", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/figcaption} mdn} *)
let figcaption' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "figcaption" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</figcaption>")

(** tag "figure", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/figure} mdn} *)
let figure : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "figure" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</figure>")

(** tag "figure", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/figure} mdn} *)
let figure' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "figure" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</figure>")

(** tag "footer", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/footer} mdn} *)
let footer : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "footer" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</footer>")

(** tag "footer", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/footer} mdn} *)
let footer' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "footer" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</footer>")

(** tag "form", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form} mdn} *)
let form : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "form" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</form>")

(** tag "form", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form} mdn} *)
let form' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "form" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</form>")

(** tag "h1", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h1} mdn} *)
let h1 : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h1" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</h1>")

(** tag "h1", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h1} mdn} *)
let h1' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h1" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</h1>")

(** tag "h2", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h2} mdn} *)
let h2 : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h2" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</h2>")

(** tag "h2", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h2} mdn} *)
let h2' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h2" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</h2>")

(** tag "h3", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h3} mdn} *)
let h3 : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h3" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</h3>")

(** tag "h3", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h3} mdn} *)
let h3' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h3" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</h3>")

(** tag "h4", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h4} mdn} *)
let h4 : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h4" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</h4>")

(** tag "h4", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h4} mdn} *)
let h4' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h4" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</h4>")

(** tag "h5", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h5} mdn} *)
let h5 : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h5" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</h5>")

(** tag "h5", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h5} mdn} *)
let h5' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h5" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</h5>")

(** tag "h6", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h6} mdn} *)
let h6 : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h6" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</h6>")

(** tag "h6", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h6} mdn} *)
let h6' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "h6" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</h6>")

(** tag "head", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/head} mdn} *)
let head : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "head" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</head>")

(** tag "head", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/head} mdn} *)
let head' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "head" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</head>")

(** tag "header", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/header} mdn} *)
let header : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "header" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</header>")

(** tag "header", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/header} mdn} *)
let header' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "header" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</header>")

(** tag "hgroup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hgroup} mdn} *)
let hgroup : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "hgroup" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</hgroup>")

(** tag "hgroup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hgroup} mdn} *)
let hgroup' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "hgroup" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</hgroup>")

(** tag "html", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/html} mdn} *)
let html : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "html" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</html>")

(** tag "html", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/html} mdn} *)
let html' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "html" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</html>")

(** tag "i", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/i} mdn} *)
let i : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "i" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</i>")

(** tag "i", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/i} mdn} *)
let i' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "i" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</i>")

(** tag "iframe", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe} mdn} *)
let iframe : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "iframe" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</iframe>")

(** tag "iframe", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe} mdn} *)
let iframe' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "iframe" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</iframe>")

(** tag "ins", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ins} mdn} *)
let ins : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ins" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</ins>")

(** tag "ins", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ins} mdn} *)
let ins' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ins" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</ins>")

(** tag "kbd", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/kbd} mdn} *)
let kbd : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "kbd" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</kbd>")

(** tag "kbd", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/kbd} mdn} *)
let kbd' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "kbd" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</kbd>")

(** tag "label", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/label} mdn} *)
let label : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "label" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</label>")

(** tag "label", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/label} mdn} *)
let label' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "label" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</label>")

(** tag "legend", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/legend} mdn} *)
let legend : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "legend" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</legend>")

(** tag "legend", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/legend} mdn} *)
let legend' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "legend" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</legend>")

(** tag "li", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/li} mdn} *)
let li : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "li" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</li>")

(** tag "li", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/li} mdn} *)
let li' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "li" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</li>")

(** tag "main", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/main} mdn} *)
let main : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "main" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</main>")

(** tag "main", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/main} mdn} *)
let main' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "main" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</main>")

(** tag "map", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/map} mdn} *)
let map : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "map" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</map>")

(** tag "map", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/map} mdn} *)
let map' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "map" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</map>")

(** tag "mark", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/mark} mdn} *)
let mark : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "mark" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</mark>")

(** tag "mark", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/mark} mdn} *)
let mark' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "mark" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</mark>")

(** tag "math", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/math} mdn} *)
let math : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "math" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</math>")

(** tag "math", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/math} mdn} *)
let math' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "math" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</math>")

(** tag "menu", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/menu} mdn} *)
let menu : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "menu" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</menu>")

(** tag "menu", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/menu} mdn} *)
let menu' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "menu" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</menu>")

(** tag "meter", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter} mdn} *)
let meter : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "meter" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</meter>")

(** tag "meter", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter} mdn} *)
let meter' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "meter" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</meter>")

(** tag "nav", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/nav} mdn} *)
let nav : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "nav" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</nav>")

(** tag "nav", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/nav} mdn} *)
let nav' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "nav" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</nav>")

(** tag "noscript", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/noscript} mdn} *)
let noscript : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "noscript" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</noscript>")

(** tag "noscript", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/noscript} mdn} *)
let noscript' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "noscript" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</noscript>")

(** tag "object", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object} mdn} *)
let object_ : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "object" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</object>")

(** tag "object", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object} mdn} *)
let object_' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "object" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</object>")

(** tag "ol", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ol} mdn} *)
let ol : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ol" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</ol>")

(** tag "ol", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ol} mdn} *)
let ol' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ol" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</ol>")

(** tag "optgroup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/optgroup} mdn} *)
let optgroup : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "optgroup" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</optgroup>")

(** tag "optgroup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/optgroup} mdn} *)
let optgroup' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "optgroup" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</optgroup>")

(** tag "option", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/option} mdn} *)
let option : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "option" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</option>")

(** tag "option", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/option} mdn} *)
let option' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "option" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</option>")

(** tag "output", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/output} mdn} *)
let output : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "output" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</output>")

(** tag "output", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/output} mdn} *)
let output' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "output" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</output>")

(** tag "p", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/p} mdn} *)
let p : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "p" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</p>")

(** tag "p", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/p} mdn} *)
let p' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "p" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</p>")

(** tag "picture", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/picture} mdn} *)
let picture : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "picture" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</picture>")

(** tag "picture", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/picture} mdn} *)
let picture' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "picture" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</picture>")

(** tag "pre", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/pre} mdn} *)
let pre : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
  Out.with_no_format_nl out @@ fun () ->
    _write_tag_attrs ~void:false out "pre" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</pre>")

(** tag "pre", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/pre} mdn} *)
let pre' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
  Out.with_no_format_nl out @@ fun () ->
    _write_tag_attrs ~void:false out "pre" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</pre>")

(** tag "progress", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/progress} mdn} *)
let progress : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "progress" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</progress>")

(** tag "progress", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/progress} mdn} *)
let progress' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "progress" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</progress>")

(** tag "q", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/q} mdn} *)
let q : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "q" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</q>")

(** tag "q", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/q} mdn} *)
let q' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "q" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</q>")

(** tag "rb", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rb} mdn} *)
let rb : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rb" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</rb>")

(** tag "rb", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rb} mdn} *)
let rb' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rb" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</rb>")

(** tag "rp", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rp} mdn} *)
let rp : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rp" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</rp>")

(** tag "rp", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rp} mdn} *)
let rp' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rp" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</rp>")

(** tag "rt", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rt} mdn} *)
let rt : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rt" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</rt>")

(** tag "rt", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rt} mdn} *)
let rt' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rt" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</rt>")

(** tag "rtc", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rtc} mdn} *)
let rtc : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rtc" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</rtc>")

(** tag "rtc", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rtc} mdn} *)
let rtc' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "rtc" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</rtc>")

(** tag "ruby", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ruby} mdn} *)
let ruby : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ruby" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</ruby>")

(** tag "ruby", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ruby} mdn} *)
let ruby' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ruby" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</ruby>")

(** tag "s", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/s} mdn} *)
let s : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "s" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</s>")

(** tag "s", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/s} mdn} *)
let s' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "s" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</s>")

(** tag "samp", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/samp} mdn} *)
let samp : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "samp" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</samp>")

(** tag "samp", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/samp} mdn} *)
let samp' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "samp" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</samp>")

(** tag "script", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script} mdn} *)
let script : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "script" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</script>")

(** tag "script", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script} mdn} *)
let script' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "script" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</script>")

(** tag "section", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/section} mdn} *)
let section : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "section" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</section>")

(** tag "section", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/section} mdn} *)
let section' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "section" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</section>")

(** tag "select", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select} mdn} *)
let select : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "select" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</select>")

(** tag "select", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select} mdn} *)
let select' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "select" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</select>")

(** tag "slot", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/slot} mdn} *)
let slot : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "slot" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</slot>")

(** tag "slot", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/slot} mdn} *)
let slot' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "slot" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</slot>")

(** tag "small", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/small} mdn} *)
let small : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "small" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</small>")

(** tag "small", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/small} mdn} *)
let small' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "small" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</small>")

(** tag "span", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/span} mdn} *)
let span : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "span" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</span>")

(** tag "span", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/span} mdn} *)
let span' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "span" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</span>")

(** tag "strong", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/strong} mdn} *)
let strong : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "strong" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</strong>")

(** tag "strong", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/strong} mdn} *)
let strong' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "strong" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</strong>")

(** tag "style", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/style} mdn} *)
let style : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "style" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</style>")

(** tag "style", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/style} mdn} *)
let style' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "style" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</style>")

(** tag "sub", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/sub} mdn} *)
let sub : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "sub" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</sub>")

(** tag "sub", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/sub} mdn} *)
let sub' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "sub" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</sub>")

(** tag "summary", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/summary} mdn} *)
let summary : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "summary" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</summary>")

(** tag "summary", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/summary} mdn} *)
let summary' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "summary" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</summary>")

(** tag "sup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/sup} mdn} *)
let sup : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "sup" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</sup>")

(** tag "sup", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/sup} mdn} *)
let sup' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "sup" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</sup>")

(** tag "svg", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/svg} mdn} *)
let svg : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "svg" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</svg>")

(** tag "svg", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/svg} mdn} *)
let svg' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "svg" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</svg>")

(** tag "table", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table} mdn} *)
let table : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "table" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</table>")

(** tag "table", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table} mdn} *)
let table' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "table" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</table>")

(** tag "tbody", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tbody} mdn} *)
let tbody : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "tbody" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</tbody>")

(** tag "tbody", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tbody} mdn} *)
let tbody' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "tbody" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</tbody>")

(** tag "td", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td} mdn} *)
let td : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "td" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</td>")

(** tag "td", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td} mdn} *)
let td' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "td" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</td>")

(** tag "template", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/template} mdn} *)
let template : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "template" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</template>")

(** tag "template", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/template} mdn} *)
let template' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "template" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</template>")

(** tag "textarea", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea} mdn} *)
let textarea : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "textarea" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</textarea>")

(** tag "textarea", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea} mdn} *)
let textarea' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "textarea" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</textarea>")

(** tag "tfoot", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tfoot} mdn} *)
let tfoot : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "tfoot" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</tfoot>")

(** tag "tfoot", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tfoot} mdn} *)
let tfoot' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "tfoot" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</tfoot>")

(** tag "th", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th} mdn} *)
let th : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "th" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</th>")

(** tag "th", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th} mdn} *)
let th' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "th" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</th>")

(** tag "thead", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/thead} mdn} *)
let thead : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "thead" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</thead>")

(** tag "thead", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/thead} mdn} *)
let thead' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "thead" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</thead>")

(** tag "time", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/time} mdn} *)
let time : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "time" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</time>")

(** tag "time", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/time} mdn} *)
let time' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "time" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</time>")

(** tag "title", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/title} mdn} *)
let title : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "title" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</title>")

(** tag "title", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/title} mdn} *)
let title' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "title" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</title>")

(** tag "tr", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tr} mdn} *)
let tr : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "tr" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</tr>")

(** tag "tr", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tr} mdn} *)
let tr' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "tr" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</tr>")

(** tag "u", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/u} mdn} *)
let u : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "u" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</u>")

(** tag "u", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/u} mdn} *)
let u' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "u" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</u>")

(** tag "ul", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ul} mdn} *)
let ul : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ul" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</ul>")

(** tag "ul", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ul} mdn} *)
let ul' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "ul" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</ul>")

(** tag "var", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/var} mdn} *)
let var : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "var" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</var>")

(** tag "var", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/var} mdn} *)
let var' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "var" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</var>")

(** tag "video", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video} mdn} *)
let video : nary = fun ?(if_=true) attrs sub out ->
  if if_ then (
    _write_tag_attrs ~void:false out "video" attrs;
    List.iter (fun sub -> Out.add_format_nl out; sub out) sub;
    if sub <> [] then Out.add_format_nl out;
    Out.add_string out "</video>")

(** tag "video", see {{:https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video} mdn} *)
let video' : nary' = fun ?(if_=true) attrs l out ->
  if if_ then (
    _write_tag_attrs ~void:false out "video" attrs;
    let has_sub = _write_subs out l in
    if has_sub then Out.add_format_nl out;
    Out.add_string out "</video>")

(** Attributes.

This module contains combinator for the standard attributes.
One can also just use a pair of strings.  *)
module A = struct
  type t = string -> attribute
  (** Attribute builder *)

  (** Attribute "accept". *)
  let accept : t = fun v -> "accept", v

  (** Attribute "accept-charset". *)
  let accept_charset : t = fun v -> "accept-charset", v

  (** Attribute "accesskey". *)
  let accesskey : t = fun v -> "accesskey", v

  (** Attribute "action". *)
  let action : t = fun v -> "action", v

  (** Attribute "align". *)
  let align : t = fun v -> "align", v

  (** Attribute "allow". *)
  let allow : t = fun v -> "allow", v

  (** Attribute "alt". *)
  let alt : t = fun v -> "alt", v

  (** Attribute "async". *)
  let async : t = fun v -> "async", v

  (** Attribute "autocapitalize". *)
  let autocapitalize : t = fun v -> "autocapitalize", v

  (** Attribute "autocomplete". *)
  let autocomplete : t = fun v -> "autocomplete", v

  (** Attribute "autofocus". *)
  let autofocus : t = fun v -> "autofocus", v

  (** Attribute "autoplay". *)
  let autoplay : t = fun v -> "autoplay", v

  (** Attribute "buffered". *)
  let buffered : t = fun v -> "buffered", v

  (** Attribute "capture". *)
  let capture : t = fun v -> "capture", v

  (** Attribute "challenge". *)
  let challenge : t = fun v -> "challenge", v

  (** Attribute "charset". *)
  let charset : t = fun v -> "charset", v

  (** Attribute "checked". *)
  let checked : t = fun v -> "checked", v

  (** Attribute "cite". *)
  let cite : t = fun v -> "cite", v

  (** Attribute "class". *)
  let class_ : t = fun v -> "class", v

  (** Attribute "code". *)
  let code : t = fun v -> "code", v

  (** Attribute "codebase". *)
  let codebase : t = fun v -> "codebase", v

  (** Attribute "cols". *)
  let cols : t = fun v -> "cols", v

  (** Attribute "colspan". *)
  let colspan : t = fun v -> "colspan", v

  (** Attribute "content". *)
  let content : t = fun v -> "content", v

  (** Attribute "contenteditable". *)
  let contenteditable : t = fun v -> "contenteditable", v

  (** Attribute "contextmenu". *)
  let contextmenu : t = fun v -> "contextmenu", v

  (** Attribute "controls". *)
  let controls : t = fun v -> "controls", v

  (** Attribute "coords". *)
  let coords : t = fun v -> "coords", v

  (** Attribute "crossorigin". *)
  let crossorigin : t = fun v -> "crossorigin", v

  (** Attribute "csp". *)
  let csp : t = fun v -> "csp", v

  (** Attribute "data". *)
  let data : t = fun v -> "data", v

  (** Attribute "data-*". *)
  let data_star : t = fun v -> "data-*", v

  (** Attribute "datetime". *)
  let datetime : t = fun v -> "datetime", v

  (** Attribute "decoding". *)
  let decoding : t = fun v -> "decoding", v

  (** Attribute "default". *)
  let default : t = fun v -> "default", v

  (** Attribute "defer". *)
  let defer : t = fun v -> "defer", v

  (** Attribute "dir". *)
  let dir : t = fun v -> "dir", v

  (** Attribute "dirname". *)
  let dirname : t = fun v -> "dirname", v

  (** Attribute "disabled". *)
  let disabled : t = fun v -> "disabled", v

  (** Attribute "download". *)
  let download : t = fun v -> "download", v

  (** Attribute "draggable". *)
  let draggable : t = fun v -> "draggable", v

  (** Attribute "enctype". *)
  let enctype : t = fun v -> "enctype", v

  (** Attribute "enterkeyhint". *)
  let enterkeyhint : t = fun v -> "enterkeyhint", v

  (** Attribute "for". *)
  let for_ : t = fun v -> "for", v

  (** Attribute "form". *)
  let form : t = fun v -> "form", v

  (** Attribute "formaction". *)
  let formaction : t = fun v -> "formaction", v

  (** Attribute "formenctype". *)
  let formenctype : t = fun v -> "formenctype", v

  (** Attribute "formmethod". *)
  let formmethod : t = fun v -> "formmethod", v

  (** Attribute "formnovalidate". *)
  let formnovalidate : t = fun v -> "formnovalidate", v

  (** Attribute "formtarget". *)
  let formtarget : t = fun v -> "formtarget", v

  (** Attribute "headers". *)
  let headers : t = fun v -> "headers", v

  (** Attribute "hidden". *)
  let hidden : t = fun v -> "hidden", v

  (** Attribute "high". *)
  let high : t = fun v -> "high", v

  (** Attribute "href". *)
  let href : t = fun v -> "href", v

  (** Attribute "hreflang". *)
  let hreflang : t = fun v -> "hreflang", v

  (** Attribute "http-equiv". *)
  let http_equiv : t = fun v -> "http-equiv", v

  (** Attribute "icon". *)
  let icon : t = fun v -> "icon", v

  (** Attribute "id". *)
  let id : t = fun v -> "id", v

  (** Attribute "importance". *)
  let importance : t = fun v -> "importance", v

  (** Attribute "integrity". *)
  let integrity : t = fun v -> "integrity", v

  (** Attribute "ismap". *)
  let ismap : t = fun v -> "ismap", v

  (** Attribute "itemprop". *)
  let itemprop : t = fun v -> "itemprop", v

  (** Attribute "keytype". *)
  let keytype : t = fun v -> "keytype", v

  (** Attribute "kind". *)
  let kind : t = fun v -> "kind", v

  (** Attribute "label". *)
  let label : t = fun v -> "label", v

  (** Attribute "lang". *)
  let lang : t = fun v -> "lang", v

  (** Attribute "language". *)
  let language : t = fun v -> "language", v

  (** Attribute "list". *)
  let list : t = fun v -> "list", v

  (** Attribute "loop". *)
  let loop : t = fun v -> "loop", v

  (** Attribute "low". *)
  let low : t = fun v -> "low", v

  (** Attribute "manifest". *)
  let manifest : t = fun v -> "manifest", v

  (** Attribute "max". *)
  let max : t = fun v -> "max", v

  (** Attribute "maxlength". *)
  let maxlength : t = fun v -> "maxlength", v

  (** Attribute "minlength". *)
  let minlength : t = fun v -> "minlength", v

  (** Attribute "media". *)
  let media : t = fun v -> "media", v

  (** Attribute "method". *)
  let method_ : t = fun v -> "method", v

  (** Attribute "min". *)
  let min : t = fun v -> "min", v

  (** Attribute "multiple". *)
  let multiple : t = fun v -> "multiple", v

  (** Attribute "muted". *)
  let muted : t = fun v -> "muted", v

  (** Attribute "name". *)
  let name : t = fun v -> "name", v

  (** Attribute "novalidate". *)
  let novalidate : t = fun v -> "novalidate", v

  (** Attribute "open". *)
  let open_ : t = fun v -> "open", v

  (** Attribute "optimum". *)
  let optimum : t = fun v -> "optimum", v

  (** Attribute "pattern". *)
  let pattern : t = fun v -> "pattern", v

  (** Attribute "ping". *)
  let ping : t = fun v -> "ping", v

  (** Attribute "placeholder". *)
  let placeholder : t = fun v -> "placeholder", v

  (** Attribute "poster". *)
  let poster : t = fun v -> "poster", v

  (** Attribute "preload". *)
  let preload : t = fun v -> "preload", v

  (** Attribute "radiogroup". *)
  let radiogroup : t = fun v -> "radiogroup", v

  (** Attribute "readonly". *)
  let readonly : t = fun v -> "readonly", v

  (** Attribute "referrerpolicy". *)
  let referrerpolicy : t = fun v -> "referrerpolicy", v

  (** Attribute "rel". *)
  let rel : t = fun v -> "rel", v

  (** Attribute "required". *)
  let required : t = fun v -> "required", v

  (** Attribute "reversed". *)
  let reversed : t = fun v -> "reversed", v

  (** Attribute "rows". *)
  let rows : t = fun v -> "rows", v

  (** Attribute "rowspan". *)
  let rowspan : t = fun v -> "rowspan", v

  (** Attribute "sandbox". *)
  let sandbox : t = fun v -> "sandbox", v

  (** Attribute "scope". *)
  let scope : t = fun v -> "scope", v

  (** Attribute "scoped". *)
  let scoped : t = fun v -> "scoped", v

  (** Attribute "selected". *)
  let selected : t = fun v -> "selected", v

  (** Attribute "shape". *)
  let shape : t = fun v -> "shape", v

  (** Attribute "size". *)
  let size : t = fun v -> "size", v

  (** Attribute "sizes". *)
  let sizes : t = fun v -> "sizes", v

  (** Attribute "slot". *)
  let slot : t = fun v -> "slot", v

  (** Attribute "span". *)
  let span : t = fun v -> "span", v

  (** Attribute "spellcheck". *)
  let spellcheck : t = fun v -> "spellcheck", v

  (** Attribute "src". *)
  let src : t = fun v -> "src", v

  (** Attribute "srcdoc". *)
  let srcdoc : t = fun v -> "srcdoc", v

  (** Attribute "srclang". *)
  let srclang : t = fun v -> "srclang", v

  (** Attribute "srcset". *)
  let srcset : t = fun v -> "srcset", v

  (** Attribute "start". *)
  let start : t = fun v -> "start", v

  (** Attribute "step". *)
  let step : t = fun v -> "step", v

  (** Attribute "style". *)
  let style : t = fun v -> "style", v

  (** Attribute "summary". *)
  let summary : t = fun v -> "summary", v

  (** Attribute "tabindex". *)
  let tabindex : t = fun v -> "tabindex", v

  (** Attribute "target". *)
  let target : t = fun v -> "target", v

  (** Attribute "title". *)
  let title : t = fun v -> "title", v

  (** Attribute "translate". *)
  let translate : t = fun v -> "translate", v

  (** Attribute "Text". *)
  let text : t = fun v -> "Text", v

  (** Attribute "type". *)
  let type_ : t = fun v -> "type", v

  (** Attribute "usemap". *)
  let usemap : t = fun v -> "usemap", v

  (** Attribute "value". *)
  let value : t = fun v -> "value", v

  (** Attribute "width". *)
  let width : t = fun v -> "width", v

  (** Attribute "wrap". *)
  let wrap : t = fun v -> "wrap", v

end
